<?php

/**
 * Unit Tests for the TeleconferenceRoomController class
 *
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2022 Comlink Inc <https://comlinkinc.com/>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace Comlink\OpenEMR\Modules\TeleHealthModule;

use Comlink\OpenEMR\Modules\TeleHealthModule\Controller\TeleconferenceRoomController;
use OpenEMR\Services\AppointmentService;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;
use Twig\Environment;

class TeleconferenceRoomControllerTest extends TestCase
{
    /**
     * @var TeleconferenceRoomController
     */
    private $controller;

    protected function setUp(): void
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        $this->controller = new TeleconferenceRoomController(
            $this->createMock(Environment::class),
            $this->createMock(LoggerInterface::class),
            ''
        );
    }

    private function getDefaultAppointment()
    {
        return [
            'pc_eid' => 1,
            'pc_catid' => 16
            ,'pc_duration' => 900
            ,'pc_hometext' => 'comlink-test-appt'
            ,'pc_apptstatus' => '*'
            ,'pc_eventDate' => date('Y-m-d')
            ,'pc_startTime' => '09:00'
            ,'pc_facility' => 3
            ,'pc_billing_location' => 3
            ,'pc_aid' => 5
            ,'pid' => 1
        ];
    }

    public function testInitalizeAppointmentForTelehealthWithPendingCheckinStatus()
    {
        $apptService = $this->createMock(AppointmentService::class);
        $appointment = $this->getDefaultAppointment();
        $appointment['pc_apptstatus'] = '^'; // pending

        $apptService->expects($this->once())
            ->method('getAppointment')
            ->willReturn($appointment);

        $this->expectException(\InvalidArgumentException::class);
        $controller = $this->controller;
        $controller->setAppointmentService($apptService);
        $controller->initalizeAppointmentForTelehealth($appointment['pc_eid']);
    }

    public function testInitalizeAppointmentForTelehealthWithValidCheckinStatus()
    {
        $encounterId = 50;
        $apptService = $this->createMock(AppointmentService::class);
        $appointment = $this->getDefaultAppointment();
        $apptService->expects($this->once())
            ->method('getAppointment')
            ->willReturn($appointment);

        $apptService->expects($this->once())
            ->method('isCheckInStatus')
            ->willReturn(false);

        $apptService->expects($this->once())
            ->method('isCheckOutStatus')
            ->willReturn(false);

        $apptService->expects($this->once())
            ->method('updateAppointmentStatus');

        $apptService->expects($this->once())
            ->method('getEncounterForAppointment')
            ->willReturn($encounterId);

        $controller = $this->controller;
        $controller->setAppointmentService($apptService);

        $updatedAppointment = $controller->initalizeAppointmentForTelehealth($appointment['pc_eid']);
        $this->assertNotEmpty($updatedAppointment, "Should return valid appointment");
        $this->assertArrayHasKey('encounter', $updatedAppointment, 'Encounter should be populated');
        $this->assertEquals($encounterId, $updatedAppointment['encounter'], 'Encounter value should match getEncounterForAppointment return');
    }

    public function testInitalizeAppointmentForTelehealthWithAppointmentCreation()
    {
        $encounterId = 50;
        $apptService = $this->createMock(AppointmentService::class);
        $appointment = $this->getDefaultAppointment();
        $apptService->expects($this->once())
            ->method('getAppointment')
            ->willReturn($appointment);

        $apptService->expects($this->once())
            ->method('isCheckInStatus')
            ->willReturn(false);

        $apptService->expects($this->once())
            ->method('isCheckOutStatus')
            ->willReturn(false);

        $apptService->expects($this->once())
            ->method('updateAppointmentStatus');

        $apptService->expects($this->once())
            ->method('getEncounterForAppointment')
            ->willReturn(null);

        $apptService->expects(($this->once()))
            ->method('createEncounterForAppointment')
            ->willReturn($encounterId);

        $controller = $this->controller;
        $controller->setAppointmentService($apptService);

        $updatedAppointment = $controller->initalizeAppointmentForTelehealth($appointment['pc_eid']);
        $this->assertNotEmpty($updatedAppointment, "Should return valid appointment");
        $this->assertArrayHasKey('encounter', $updatedAppointment, 'Encounter should be populated');
        $this->assertEquals($encounterId, $updatedAppointment['encounter'], 'Encounter value should match createEncounterForAppointment return');
    }
}
