<?php

/**
 * GeoTelemetry class
 *
 * Over 80% of GeoTelemetry class code was generated by ChatGPT code copilot.
 * The rest was modified to fit the OpenEMR coding style.
 *
 * Lightweight geolocation and anonymization utility
 * - Uses public IP lookup APIs
 * - Composer-free
 * - Fallback support and IP anonymization
 *
 * @package        OpenEMR
 * @link           https://www.open-emr.org
 * @author         Jerry Padgett <sjpadgett@gmail.com>
 * @copyright      Copyright (c) 2025 <sjpadgett@gmail.com>
 * @license        https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace OpenEMR\Telemetry;

class GeoTelemetry
{
    /**
     * Anonymize IP using SHA-256 hashing
     */
    public function anonymizeIp(string $ip): string
    {
        return hash('sha256', $ip);
    }

    /**
     * Get geolocation data from IP using a lightweight external API
     */
    public function getGeoData(string $ip): array
    {
        $data = $this->fetch("https://ipapi.co/{$ip}/json/");
        if (isset($data['country_name'])) {
            return [
                'country' => $data['country_name'] ?? null,
                'region' => $data['region'] ?? null,
                'city' => $data['city'] ?? null,
                'latitude' => $data['latitude'] ?? null,
                'longitude' => $data['longitude'] ?? null,
            ];
        }
        // fallback to geoplugin.net
        $data = $this->fetch("http://www.geoplugin.net/json.gp?ip={$ip}");
        if (isset($data['geoplugin_countryName'])) {
            return [
                'country' => $data['geoplugin_countryName'] ?? null,
                'region' => $data['geoplugin_region'] ?? null,
                'city' => $data['geoplugin_city'] ?? null,
                'latitude' => $data['geoplugin_latitude'] ?? null,
                'longitude' => $data['geoplugin_longitude'] ?? null,
            ];
        }

        return ['error' => 'IP lookup failed'];
    }

    /**
     * Get geolocation of the current server (public-facing IP)
     */
    public function getServerGeoData(): array
    {
        $ip = trim(@file_get_contents("https://api.ipify.org"));
        if (filter_var($ip, FILTER_VALIDATE_IP)) {
            return $this->getGeoData($ip);
        }
        return ['error' => 'Unable to determine server IP'];
    }

    /**
     * Get geolocation data from a given URL
     *
     * @param string $url
     * @return array
     */
    private function fetch(string $url): array
    {
        $ctx = stream_context_create(['http' => ['timeout' => 3]]);
        $json = @file_get_contents($url, false, $ctx);
        return json_decode($json, true) ?: [];
    }
}
